"""
缓存服务层：封装所有缓存相关操作
"""
import logging

from meraki_Interface_forward.redis_utils import (
    set_json,
    CacheKey,
    get_redis_client,
)

logger = logging.getLogger("meraki_Interface_forward.services.cache_service")

# 单设备缓存前缀
DEVICE_CACHE_PREFIX = "device:"
UPLINK_CACHE_PREFIX = "uplink:"
STATUS_CACHE_PREFIX = "status:"
CHANNEL_UTILIZATION_CACHE_PREFIX = "channel_utilization:"


def cache_devices_by_serial(devices, ttl: int = 60 * 60 * 12) -> None:
    """
    将 Meraki 设备列表按 serial 维度拆分，批量写入 Redis。
    使用 Pipeline 优化，避免 N+1 次网络请求。
    """
    if not isinstance(devices, list) or not devices:
        return
    
    client = get_redis_client()
    if not client:
        return

    try:
        pipeline = client.pipeline()
        import json
        for dev in devices:
            if not isinstance(dev, dict):
                continue
            serial = dev.get("serial")
            if not serial:
                continue
            
            # 直接使用 pipeline.set，手动序列化 json
            # 注意：set_json 内部是调用的 client.set，这里为了 pipeline 效率直接操作
            key = f"{DEVICE_CACHE_PREFIX}{serial}"
            val = json.dumps(dev, ensure_ascii=False)
            pipeline.set(key, val, ex=ttl)
            
        pipeline.execute()
    except Exception as e:
        logger.error(f"cache_devices_by_serial pipeline error: {e}")



def cache_uplinks_by_serial(uplinks, ttl: int = 60 * 60 * 6) -> None:
    """将组织级 uplink 列表按 serial 拆分缓存"""
    if not isinstance(uplinks, list):
        return
    for item in uplinks:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        try:
            set_json(f"{UPLINK_CACHE_PREFIX}{serial}", item, ex=ttl)
        except Exception:
            continue


def cache_status_by_serial(status_list, ttl: int = 65) -> None:
    """
    将组织级设备状态列表按 serial 拆分缓存，只保存 status 字段值
    - key: status:<serial>
    - value: 设备的 status 字段值（字符串，如 "online", "offline" 等）
    """
    if not isinstance(status_list, list):
        return
    for item in status_list:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        status_val = item.get("status")
        try:
            set_json(f"{STATUS_CACHE_PREFIX}{serial}", status_val, ex=ttl)
        except Exception:
            continue


def cache_channel_utilization_by_serial(utilization_list, ttl: int = 300) -> None:
    """将组织级信道利用率列表按 serial 拆分缓存"""
    if not isinstance(utilization_list, list):
        return
    for item in utilization_list:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        try:
            set_json(f"{CHANNEL_UTILIZATION_CACHE_PREFIX}{serial}", item, ex=ttl)
        except Exception:
            continue

