"""
Meraki API 服务层：封装所有 Meraki API 调用
"""
import os
import meraki
import logging
import json

from meraki_Interface_forward.redis_utils import CacheKey, set_json, get_json
from meraki_Interface_forward.services.cache_service import cache_devices_by_serial

logger = logging.getLogger("meraki_Interface_forward.services.meraki_service")

# Meraki API 配置
API_KEY = os.getenv("MERAKI_API_KEY")
ORGANIZATION_ID = os.getenv("ORGANIZATION_ID")
MERAKI_API_BASE = os.getenv("MERAKI_API_BASE")

dashboard = meraki.DashboardAPI(
    api_key=API_KEY,
    base_url=MERAKI_API_BASE,
    output_log=True,
    print_console=True,
    suppress_logging=True,
    wait_on_rate_limit=True,
    nginx_429_retry_wait_time=5,
    maximum_retries=3,
)


def get_organization_devices():
    """获取组织所有设备"""
    return dashboard.organizations.getOrganizationDevices(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all'
    )


def get_organization_networks():
    """获取组织所有网络"""
    try:
        networks = dashboard.organizations.getOrganizationNetworks(
            ORGANIZATION_ID,
            perPage=1000,
            total_pages='all'
        )
        if len(networks):
            set_json(CacheKey.NETWORKS.value, networks)
            return networks
    except Exception as e:
        msg = str(e)
        if "Expecting value: line 1 column 1" in msg:
            return []
        logger.error(f"获取组织网络失败: {e}")
        return None
    return None


def get_organization_status():
    """获取组织所有设备状态"""
    return dashboard.organizations.getOrganizationDevicesAvailabilities(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all'
    )


def get_organization_status_overview():
    """获取组织所有设备状态概览"""
    return dashboard.organizations.getOrganizationDevicesStatusesOverview(ORGANIZATION_ID)


def get_organization_alert():
    """获取组织所有设备告警"""
    alerts = dashboard.organizations.getOrganizationAssuranceAlerts(
        ORGANIZATION_ID, total_pages="all"
    )

    processed = []
    if alerts:
        for item in alerts or []:
            scope = item.get("scope", {})
            devices = scope.get("devices", [])

            # 有多个设备 → 每个设备都生成一条数据
            if devices:
                for dev in devices:
                    processed.append({
                        "id": item.get("id"),
                        "categoryType": item.get("categoryType"),
                        "networkId": item.get("network", {}).get("id"),
                        "networkName": item.get("network", {}).get("name"),
                        "type": item.get("type"),
                        "title": item.get("title"),
                        "severity": item.get("severity"),
                        "deviceType": item.get("deviceType"),
                        "startedAt": item.get("startedAt"),
                        "resolvedAt": item.get("resolvedAt"),
                        "dismissedAt": item.get("dismissedAt"),
                        # 多设备的关键字段
                        "serial": dev.get("serial"),
                        "mac": dev.get("mac"),
                        "deviceName": dev.get("name"),
                        "port": dev.get("portIdentifier"),
                        "scope": scope
                    })
    return processed


def get_organization_uplinks():
    """获取组织所有设备 uplink"""
    return dashboard.organizations.getOrganizationDevicesUplinksAddressesByDevice(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all',
    )


def get_organization_channel_utilization():
    """获取组织所有设备信道利用率"""
    return dashboard.wireless.getOrganizationWirelessDevicesChannelUtilizationByDevice(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all',
        timespan=3600
    )


def get_device_switch_ports_status(serial):
    """获取单台交换机端口状态"""
    return dashboard.switch.getDeviceSwitchPortsStatuses(
        serial,
        timespan=3600,
    )


def get_filtered_devices(filter_networks=None):
    """
    获取经过过滤和处理的设备列表
    :param filter_networks: list of network names to exclude
    :return: list of device dicts
    """
    filter_names = set(filter_networks) if isinstance(filter_networks, list) else set()

    # 获取网络列表（缓存优先）
    networks = get_json(CacheKey.NETWORKS.value) or []
    if isinstance(networks, str):
        try:
            networks = json.loads(networks)
        except Exception:
            networks = []
    
    if not networks:
        networks = get_organization_networks() or []

    network_map = {}
    if isinstance(networks, list):
         network_map = {net.get("id"): net.get("name") for net in networks if isinstance(net, dict)}

    # 网络名 -> id 映射，用于过滤
    exclude_network_ids = set()
    if filter_names and isinstance(networks, list):
        for net in networks:
            if isinstance(net, dict) and net.get("name") in filter_names:
                nid = net.get("id")
                if nid:
                    exclude_network_ids.add(nid)

    # 获取设备（缓存优先，未命中则回源）
    devices = get_json(CacheKey.DEVICES.value)
    if not devices:
        logger.info("get_filtered_devices: Redis miss for ALL devices. Fetching from Meraki API...")
        devices = get_organization_devices()
        if devices:
            set_json(CacheKey.DEVICES.value, devices, ex=60 * 60)
            logger.info(f"get_filtered_devices: Fetched {len(devices)} devices. Caching by serial...")
            cache_devices_by_serial(devices, ttl=60 * 60)
            logger.info("get_filtered_devices: Caching complete.")
    else:
        logger.info(f"get_filtered_devices: Redis hit. {len(devices)} devices found.")

    if not devices:
        return []

    # 过滤：networkId 命中需要排除的网络则丢弃
    if exclude_network_ids:
        filtered = []
        for dev in devices:
            if not isinstance(dev, dict):
                continue
            nid = dev.get("networkId")
            if nid in exclude_network_ids:
                continue
            filtered.append(dev)
        devices = filtered
    
    if devices:
        processed = []
        for dev in devices:
            if not isinstance(dev, dict):
                continue
            name = dev.get("name")
            serial = dev.get("serial")
            mac = dev.get("mac")
            if not name or (isinstance(name, str) and not name.strip()):
                dev["name"] = serial or mac or "UNKNOWN"
            
            # 追加 networkName 字段
            net_id = dev.get("networkId")
            if network_map and net_id:
                dev["networkName"] = network_map.get(net_id)

            processed.append(dev)
        
        # 重名处理
        name_counts = {}
        for dev in processed:
            n = dev.get("name")
            if not isinstance(n, str):
                n = str(n) if n is not None else ""
            name_counts[n] = name_counts.get(n, 0) + 1
        for dev in processed:
            n = dev.get("name")
            if name_counts.get(n, 0) > 1:
                s = dev.get("serial") or ""
                dev["name"] = f"{n}{s}"
        devices = processed

    return devices

