"""
缓存管理视图
"""
import logging
from django.http import JsonResponse

from meraki_Interface_forward.redis_utils import (
    get_json,
    ttl,
    delete,
    CacheKey,
    scan_delete,
    get_redis_client,
)
from meraki_Interface_forward.utils import build_prefix_stats
from meraki_Interface_forward.services.cache_service import (
    DEVICE_CACHE_PREFIX,
    UPLINK_CACHE_PREFIX,
    STATUS_CACHE_PREFIX,
    CHANNEL_UTILIZATION_CACHE_PREFIX,
)

logger = logging.getLogger("meraki_Interface_forward.views.cache_views")


def cache_get_device(request):
    """
    查询缓存健康度（极速优化版）：
    - 返回各集中缓存 key 的剩余 TTL（CacheKey.*）
    - 针对按前缀拆分的缓存（device/uplink/status/channel/switch_ports/rl/lock）
      返回数量、抽样 5 条的平均与最小 TTL，极速检查空前缀
    - 兼容返回一个示例设备数据，供快速验证
    """
    # 集中缓存 TTL（使用 pipeline 批量查询，减少网络往返）
    client = get_redis_client()
    cache_keys_ttl = {}
    if client:
        try:
            pipe = client.pipeline()
            cache_key_list = [
                CacheKey.ORGANIZATIONS.value,
                CacheKey.NETWORKS.value,
                CacheKey.DEVICES.value,
                CacheKey.DEVICES_AVAILABILITIES.value,
                CacheKey.DEVICES_STATUSES_OVERVIEW.value,
                CacheKey.DEVICES_STATUSES.value,
                CacheKey.DEVICES_UPLINKS_BY_DEVICE.value,
                CacheKey.WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE.value,
                CacheKey.ASSURANCE_ALERTS.value,
            ]
            for key in cache_key_list:
                pipe.ttl(key)
            ttl_results = pipe.execute()

            cache_keys_ttl = {
                "organizations": ttl_results[0] if len(ttl_results) > 0 else -2,
                "networks": ttl_results[1] if len(ttl_results) > 1 else -2,
                "devices": ttl_results[2] if len(ttl_results) > 2 else -2,
                "devices_availabilities": ttl_results[3] if len(ttl_results) > 3 else -2,
                "devices_statuses_overview": ttl_results[4] if len(ttl_results) > 4 else -2,
                "devices_statuses": ttl_results[5] if len(ttl_results) > 5 else -2,
                "devices_uplinks_by_device": ttl_results[6] if len(ttl_results) > 6 else -2,
                "wireless_channel_utilization_by_device": ttl_results[7] if len(ttl_results) > 7 else -2,
                "assurance_alerts": ttl_results[8] if len(ttl_results) > 8 else -2,
            }
        except Exception:
            # pipeline 失败时回退到单个查询
            cache_keys_ttl = {
                "organizations": ttl(CacheKey.ORGANIZATIONS.value),
                "networks": ttl(CacheKey.NETWORKS.value),
                "devices": ttl(CacheKey.DEVICES.value),
                "devices_availabilities": ttl(CacheKey.DEVICES_AVAILABILITIES.value),
                "devices_statuses_overview": ttl(CacheKey.DEVICES_STATUSES_OVERVIEW.value),
                "devices_statuses": ttl(CacheKey.DEVICES_STATUSES.value),
                "devices_uplinks_by_device": ttl(CacheKey.DEVICES_UPLINKS_BY_DEVICE.value),
                "wireless_channel_utilization_by_device": ttl(CacheKey.WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE.value),
                "assurance_alerts": ttl(CacheKey.ASSURANCE_ALERTS.value),
            }
    else:
        # Redis 不可用时返回默认值
        cache_keys_ttl = {k: -2 for k in [
            "organizations", "networks", "devices", "devices_availabilities",
            "devices_statuses_overview", "devices_statuses", "devices_uplinks_by_device",
            "wireless_channel_utilization_by_device", "assurance_alerts"
        ]}

    # 前缀类缓存统计（极速优化：空前缀只扫描 10 个 key）
    prefix_stats = build_prefix_stats(
        {
            "device": f"{DEVICE_CACHE_PREFIX}*",
            "uplink": f"{UPLINK_CACHE_PREFIX}*",
            "status": f"{STATUS_CACHE_PREFIX}*",
            "channel_utilization": f"{CHANNEL_UTILIZATION_CACHE_PREFIX}*",
            "switch_ports_status": "switch_ports_status:*",
            "rate_limit": "rl:*",
            "lock": "lock:*",
        }
    )

    # 兼容返回示例设备与数据有效性（只在 devices 缓存有效时获取）
    sample_device = None
    remaining_ttl = cache_keys_ttl.get("devices")
    data_is_valid = isinstance(remaining_ttl, int) and remaining_ttl > 0
    if data_is_valid:
        devices = get_json(CacheKey.DEVICES.value)
        if isinstance(devices, list) and devices:
            sample_device = devices[0]

    return JsonResponse(
        {
            "dataIsValid": bool(sample_device),
            "sampleDevice": sample_device,
            "cacheKeysTtl": cache_keys_ttl,
            "prefixStats": prefix_stats,
        },
        safe=False,
        json_dumps_params={"indent": 2, "ensure_ascii": False},
    )


def clear_cache(request):
    """
    清理 Redis 中非必要的缓存 key，主要用于测试/运维场景下清除脏数据。
    不会删除业务以外的 key，只清除本项目约定的缓存前缀和 CacheKey。
    """
    try:
        # 1. 删除基于 CacheKey 的集中缓存
        cache_keys = [
            CacheKey.ORGANIZATIONS.value,
            CacheKey.NETWORKS.value,
            CacheKey.DEVICES.value,
            CacheKey.DEVICES_AVAILABILITIES.value,
            CacheKey.DEVICES_STATUSES_OVERVIEW.value,
            CacheKey.DEVICES_STATUSES.value,
            CacheKey.DEVICES_UPLINKS_BY_DEVICE.value,
            CacheKey.WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE.value,
            CacheKey.ASSURANCE_ALERTS.value,
        ]
        deleted_cache_keys = 0
        for k in cache_keys:
            try:
                deleted_cache_keys += delete(k) or 0
            except Exception:
                continue

        # 2. 删除按 serial 维度的单设备缓存 & 限流 / 锁 key
        patterns = [
            f"{DEVICE_CACHE_PREFIX}*",
            f"{UPLINK_CACHE_PREFIX}*",
            f"{STATUS_CACHE_PREFIX}*",
            f"{CHANNEL_UTILIZATION_CACHE_PREFIX}*",
            "rl:*",        # 限流 key
            "lock:*",      # 分布式锁 key
        ]
        pattern_deleted = {}
        for p in patterns:
            count = scan_delete(p)
            pattern_deleted[p] = count

        return JsonResponse(
            {
                "message": "缓存清理完成",
                "deleted_cache_keys": deleted_cache_keys,
                "deleted_by_pattern": pattern_deleted,
            },
            safe=False,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )
    except Exception as e:
        logger.exception("clear_cache 调用失败")
        return JsonResponse({"error": str(e)}, status=500)

