import json
import logging
import requests
from meraki_Interface_forward.redis_utils import get_json, set_json

logger = logging.getLogger("meraki_Interface_forward.services.juniper_service")

# Cache keys
JUNIPER_DEVICES_CACHE_KEY = "juniper:devices"
JUNIPER_SITES_CACHE_KEY = "juniper:sites"

def fetch_juniper_devices(api_url, org_id, api_token):
    """
    Fetch devices from Juniper API and enrich with site names.
    Returns a list of device dicts.
    """
    if not api_url or not org_id or not api_token:
        logger.error("Missing required Juniper API parameters")
        return []

    # Try cache first (optional, but good for performance)
    # Since parameters might change, we might want to include org_id in cache key if multi-tenant support is needed in future.
    # For now, using a simple key as per request pattern.
    # To ensure freshness, we can skip cache or set short TTL. 
    # Given the requirement "like Meraki logic", let's fetch fresh if not specified otherwise, 
    # but we can implement basic request logic here.

    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Token {api_token}"
    }

    try:
        # 1. Fetch Sites
        sites_url = f"{api_url}/api/v1/orgs/{org_id}/sites"
        logger.info(f"Fetching Juniper sites from {sites_url}")
        resp_sites = requests.get(sites_url, headers=headers, timeout=30)
        resp_sites.raise_for_status()
        sites = resp_sites.json()
        
        site_map = {s["id"]: s["name"] for s in sites if "id" in s and "name" in s}
        logger.info(f"Fetched {len(sites)} sites.")

        # 2. Fetch Devices (Pagination handling is simplified to limit=1000 for now, 
        # or we can implement pagination if needed. User script showed limit/page logic)
        # We'll fetch a large chunk.
        limit = 3000 
        devices_url = f"{api_url}/api/v1/orgs/{org_id}/inventory?limit={limit}"
        logger.info(f"Fetching Juniper devices from {devices_url}")
        resp_devices = requests.get(devices_url, headers=headers, timeout=60)
        resp_devices.raise_for_status()
        devices = resp_devices.json()
        logger.info(f"Fetched {len(devices)} devices.")

        processed_devices = []
        for dev in devices:
            # Filter logic: only switch and ap (wireless)
            # User said "wireless device type is ap"
            dev_type = dev.get("type")
            if dev_type not in ["switch", "ap"]:
                continue
            
            # Enrich with site name
            site_id = dev.get("site_id")
            dev["site_name"] = site_map.get(site_id, "")
            
            processed_devices.append(dev)
            
        logger.info(f"Processed {len(processed_devices)} valid Juniper devices (switch/ap).")
        return processed_devices

    except Exception as e:
        logger.error(f"Error fetching Juniper data: {e}")
        return []
