"""
任务模块：数据同步任务（异步执行）
"""
import threading
import time
import logging
from django.http import JsonResponse

from meraki_Interface_forward.resultAnalysis import logger
from meraki_Interface_forward.services.meraki_service import (
    get_organization_networks,
    get_organization_status_overview,
    get_organization_status,
    get_organization_alert,
    get_organization_devices,
)
from meraki_Interface_forward.services.cache_service import cache_devices_by_serial
from meraki_Interface_forward.redis_utils import (
    set_json,
    get_json,
    CacheKey,
)

# 任务状态 key
TASK_STATUS_KEY = "task:synchronization:status"
TASK_START_TIME_KEY = "task:synchronization:start_time"
TASK_FINISH_TIME_KEY = "task:synchronization:finish_time"

# 任务状态枚举
TASK_STATUS_RUNNING = "执行中"
TASK_STATUS_COMPLETED = "已完成"
TASK_STATUS_FAILED = "执行失败"
TASK_STATUS_IDLE = "空闲"


def _execute_synchronization():
    """
    执行数据同步任务（内部函数，在后台线程中执行）
    """
    start_time = time.time()
    set_json(TASK_START_TIME_KEY, start_time, ex=3600)  # 1小时过期
    set_json(TASK_STATUS_KEY, TASK_STATUS_RUNNING, ex=3600)

    try:
        # 1. 同步网络
        try:
            networks = get_organization_networks()
            if networks:
                set_json(CacheKey.NETWORKS.value, networks)
            logger.info("网络数据同步完成")
        except Exception as e:
            logger.error(f"Failed to cache networks: {e}")

        # 2. 同步设备状态概览
        try:
            status_overview = get_organization_status_overview()
            if status_overview:
                set_json(CacheKey.DEVICES_STATUSES_OVERVIEW.value, status_overview, ex=60 * 5)
            logger.info("设备状态概览同步完成")
        except Exception as e:
            logger.error(f"Failed to cache status overview: {e}")

        # 3. 同步设备状态
        try:
            status = get_organization_status()
            if status:
                set_json(CacheKey.DEVICES_STATUSES.value, status, ex=60 * 5)
            logger.info("设备状态同步完成")
        except Exception as e:
            logger.error(f"Failed to cache status: {e}")

        # 4. 同步告警
        try:
            alerts = get_organization_alert()
            if alerts:
                set_json(CacheKey.ASSURANCE_ALERTS.value, alerts, ex= 60 * 5 )
            logger.info("告警数据同步完成")
        except Exception as e:
            logger.error(f"Failed to cache alert: {e}")

        # 5. 同步设备列表（最耗时，可能超过60秒）
        try:
            devices = get_organization_devices()
            if devices:
                set_json(CacheKey.DEVICES.value, devices, ex=60 * 60)
                # 同步按 serial 建立单设备缓存，便于高并发单设备查询
                cache_devices_by_serial(devices, ttl=60 * 60)
            logger.info(f"设备列表同步完成，共 {len(devices) if devices else 0} 台设备")
        except Exception as e:
            logger.error(f"Failed to cache devices: {e}")

        # 任务完成
        elapsed_time = time.time() - start_time
        set_json(TASK_STATUS_KEY, TASK_STATUS_COMPLETED, ex=3600)
        set_json(TASK_FINISH_TIME_KEY, time.time(), ex=3600)
        logger.info(f"数据同步任务完成，耗时 {elapsed_time:.2f} 秒")

    except Exception as e:
        logger.exception("数据同步任务执行失败")
        set_json(TASK_STATUS_KEY, TASK_STATUS_FAILED, ex=3600)
        set_json(TASK_FINISH_TIME_KEY, time.time(), ex=3600)


def synchronization_data(request):
    """
    数据同步接口（异步执行）
    
    返回当前任务状态：
    - status: 执行状态（执行中/已完成/执行失败/空闲）
    - elapsed_time: 执行时间（秒）
    - message: 状态描述
    - 支持查询模式：带 execute=1 时，只查询状态，不触发新任务
    """
    try:
        execute_only = request.GET.get("execute") == "1"

        # 检查当前任务状态
        current_status = get_json(TASK_STATUS_KEY) or TASK_STATUS_IDLE
        start_time = get_json(TASK_START_TIME_KEY)
        finish_time = get_json(TASK_FINISH_TIME_KEY)

        # 如果任务正在执行中，返回当前状态
        if current_status == TASK_STATUS_RUNNING:
            elapsed_time = None
            if start_time:
                elapsed_time = round(time.time() - start_time, 2)
            
            return JsonResponse(
                {
                    "status": TASK_STATUS_RUNNING,
                    "elapsed_time": elapsed_time,
                    "message": "数据同步任务正在执行中，请稍后查询",
                    "last_finished_at": finish_time,
                },
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 仅查询模式：不触发新任务
        if execute_only:
            elapsed_time = None
            if start_time and current_status == TASK_STATUS_RUNNING:
                elapsed_time = round(time.time() - start_time, 2)

            return JsonResponse(
                {
                    "status": current_status,
                    "elapsed_time": elapsed_time,
                    "message": f"当前任务状态: {current_status}",
                    "last_finished_at": finish_time,
                },
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 如果任务空闲/已完成/失败，启动新任务并返回“创建成功”语句，附带上次完成时间
        if current_status in [TASK_STATUS_COMPLETED, TASK_STATUS_FAILED, TASK_STATUS_IDLE]:
            thread = threading.Thread(target=_execute_synchronization, daemon=True)
            thread.start()
            
            return JsonResponse(
                {
                    "status": "已启动",
                    "elapsed_time": 0,
                    "message": "数据同步任务已创建并在后台执行",
                    "last_finished_at": finish_time,
                },
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 其他情况，返回当前状态
        elapsed_time = None
        if start_time:
            elapsed_time = round(time.time() - start_time, 2)

        return JsonResponse(
            {
                "status": current_status,
                "elapsed_time": elapsed_time,
                "message": f"当前任务状态: {current_status}",
                "last_finished_at": finish_time,
            },
            safe=False,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )

    except Exception as e:
        logger.exception("synchronization_data 调用失败")
        return JsonResponse(
            {"error": str(e), "status": TASK_STATUS_FAILED},
            status=500,
            safe=False,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )
