import random
from typing import Dict

from meraki_Interface_forward.redis_utils import get_redis_client


def collect_prefix_ttl_stats(pattern: str, sample_size: int = 10, max_scan_keys: int = 50, quick_check: bool = True) -> Dict[str, object]:
    """
    收集指定前缀的 key 数量与 TTL 统计（极速优化版）。
    - 空前缀：只扫描 10 个 key，如果没找到立即返回（< 10ms）
    - 有数据前缀：限制扫描范围，不进行全量计数，只统计已扫描的 key
    - 批量查询：使用 pipeline 批量获取 TTL
    """
    client = get_redis_client()
    if not client:
        return {"count": 0, "sampled": 0, "minTtl": None, "avgTtl": None}

    keys = []
    try:
        # 极速检查：空前缀只扫描 10 个 key，如果没找到立即返回
        cursor, batch = client.scan(cursor=0, match=pattern, count=10 if quick_check else 50)
        batch = batch or []
        if not batch:
            # 空前缀：立即返回，不继续扫描
            return {"count": 0, "sampled": 0, "minTtl": None, "avgTtl": None}
        
        keys.extend(batch)
        
        # 如果找到了数据，继续扫描但严格限制总量（不再进行全量计数）
        if cursor != 0 and len(keys) < max_scan_keys:
            # 只再扫描一次，获取更多 key 用于采样
            cursor, batch = client.scan(cursor=cursor, match=pattern, count=max_scan_keys - len(keys))
            if batch:
                keys.extend(batch[:max_scan_keys - len(keys)])
    except Exception:
        return {"count": 0, "sampled": 0, "minTtl": None, "avgTtl": None}

    if not keys:
        return {"count": 0, "sampled": 0, "minTtl": None, "avgTtl": None}

    # 限制采样数量（进一步减少 TTL 查询）
    actual_sample_size = min(len(keys), sample_size)
    sampled_keys = random.sample(keys, actual_sample_size)
    
    # 使用 pipeline 批量查询 TTL（一次性查询所有采样 key）
    ttls = []
    try:
        pipe = client.pipeline()
        for k in sampled_keys:
            pipe.ttl(k)
        results = pipe.execute()
        for t in results:
            if isinstance(t, int) and t >= 0:
                ttls.append(t)
    except Exception:
        # pipeline 失败时回退到单个查询（但这种情况应该很少）
        for k in sampled_keys[:5]:  # 只查询前 5 个，避免超时
            try:
                t = client.ttl(k)
                if isinstance(t, int) and t >= 0:
                    ttls.append(t)
            except Exception:
                continue

    # 对于 device 这种可能有大量 key 的情况，count 只返回实际扫描到的数量
    # 不再进行全量计数，避免性能问题
    return {
        "count": len(keys),  # 只返回实际扫描到的数量，不再全量计数
        "sampled": len(sampled_keys),
        "minTtl": min(ttls) if ttls else None,
        "avgTtl": round(sum(ttls) / len(ttls), 2) if ttls else None,
    }


def build_prefix_stats(prefix_patterns: Dict[str, str], sample_size: int = 10) -> Dict[str, dict]:
    """
    生成按前缀的 TTL 统计结果（极速优化版）。
    - 空前缀：极速检查（只扫描 10 个 key）
    - device 前缀：限制扫描 50 个 key，采样 5 个
    - 其他前缀：限制扫描 30 个 key，采样 5 个
    """
    result = {}
    for name, pattern in prefix_patterns.items():
        if name == "device":
            # device 前缀：扫描 50 个 key，采样 5 个
            result[name] = collect_prefix_ttl_stats(pattern, sample_size=5, max_scan_keys=50, quick_check=True)
        else:
            # 其他前缀：极速检查，扫描 30 个 key，采样 5 个
            result[name] = collect_prefix_ttl_stats(pattern, sample_size=5, max_scan_keys=30, quick_check=True)
    return result

