import json
import logging
import os
import concurrent.futures
from django.conf import settings
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt
from meraki_Interface_forward.services.juniper_service import fetch_juniper_devices
from meraki_Interface_forward.services.zabbix_service import ZabbixService
from meraki_Interface_forward.redis_utils import get_json, set_json

logger = logging.getLogger("meraki_Interface_forward.views.juniper_views")

# Redis Key for storing synced host serials (Set or List)
ZABBIX_JUNIPER_SYNCED_HOSTS_KEY = "zabbix:juniper_synced_hosts"

# Max workers for concurrent Zabbix API calls
MAX_WORKERS = 20

def get_default_juniper_group_name(dev_type):
    if dev_type == "ap": # wireless
        return "Juniper/Wireless"
    if dev_type == "switch":
        return "Juniper/Switch"
    return "Juniper/Other"

@csrf_exempt
def sync_juniper_to_zabbix(request):
    """
    Sync Juniper devices to Zabbix
    """
    if request.method != "POST":
        return JsonResponse({"message": "Method Not Allowed"}, status=405)

    try:
        logger.info("Starting Juniper to Zabbix synchronization task...")
        
        body = {}
        if request.body:
            body = json.loads(request.body.decode("utf-8"))
        
        juniper_api = body.get("juniperApi")
        org_id = body.get("orgId")
        api_token = body.get("apiToken")
        site_group_map = body.get("siteGroupMap") or {}
        disable_host_tag = body.get("disableHostTag")
        
        logger.info(f"Request parameters - OrgID: {org_id}, Site Map Keys: {list(site_group_map.keys())}")

        # 1. Get Devices
        devices = fetch_juniper_devices(juniper_api, org_id, api_token)
        
        current_device_map = {} # serial -> device
        if devices:
             for d in devices:
                 if isinstance(d, dict) and d.get("serial"):
                     current_device_map[d.get("serial")] = d
        
        current_serials = set(current_device_map.keys())
        logger.info(f"Identified {len(current_serials)} unique valid Juniper devices.")

        # 2. Init Zabbix Service
        try:
            zabbix = ZabbixService()
            zabbix_groups = zabbix.get_hostgroups() # {name: id}
        except Exception as e:
            return JsonResponse({"error": f"Zabbix connection failed: {str(e)}"}, status=500)
        
        # 3. Redis & Sync Logic
        cached_serials_list = get_json(ZABBIX_JUNIPER_SYNCED_HOSTS_KEY) or []
        cached_serials = set(cached_serials_list)
        
        to_disable = cached_serials - current_serials
        logger.info(f"Diff calculation: {len(to_disable)} hosts to disable, {len(current_serials)} hosts to add/update.")

        results = {
            "created": [],
            "skipped": [],
            "failed": [],
            "disabled": [],
            "failed_disable": []
        }

        # 3.1 Disable removed hosts
        if to_disable:
            logger.info(f"Processing {len(to_disable)} disabled hosts with {MAX_WORKERS} threads...")
            
            def _disable_task(s_serial):
                desc = "该设备从Juniper设备接口 无法发现, 设备已经从Juniper平台删除!"
                tags = None
                if disable_host_tag and isinstance(disable_host_tag, dict):
                    tags = [disable_host_tag]
                
                return zabbix.update_host_details(
                    host_name=s_serial,
                    description=desc,
                    tags=tags,
                    status=1
                )

            with concurrent.futures.ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
                future_to_serial = {executor.submit(_disable_task, s): s for s in to_disable}
                for future in concurrent.futures.as_completed(future_to_serial):
                    s = future_to_serial[future]
                    try:
                        res = future.result()
                        if res["status"] == "updated":
                            results["disabled"].append(res)
                        elif res["status"] == "not_found":
                             pass
                        else:
                            results["failed_disable"].append(res)
                    except Exception as exc:
                        logger.error(f"Disable task generated an exception for {s}: {exc}")
                        results["failed_disable"].append({"host": s, "error": str(exc)})

        # Pre-fetch Template IDs
        logger.info("Pre-fetching Zabbix template IDs...")
        template_cache = {}
        # Juniper templates + Common ICMP
        target_templates = ["Env_Juniper_Wireless", "Env_Juniper_Switch", "Env_Meraki_ICMP_Ping"]
        for tmpl_name in target_templates:
             tid = zabbix.get_template_id_by_name(tmpl_name)
             if tid:
                 template_cache[tmpl_name] = tid
             else:
                 logger.warning(f"Template not found: {tmpl_name}")

        # 3.2 Process Current Devices
        logger.info(f"Processing {len(current_device_map)} devices with {MAX_WORKERS} threads...")

        def _create_update_task(dev_info):
            serial = dev_info.get("serial")
            name = dev_info.get("name") # Or hostname? Using name as visible name
            dev_type = dev_info.get("type") # switch or ap
            site_name = dev_info.get("site_name")
            mac = dev_info.get("mac")
            dev_id = dev_info.get("id")
            site_id = dev_info.get("site_id")
            
            # Juniper doesn't explicitly provide IP in inventory endpoint usually, 
            # and the user stated that Juniper list API does NOT return interface info.
            # So we pass an empty list for interfaces.
            interfaces = []

            # Target Group
            target_group_name = None
            if site_name and site_name in site_group_map:
                target_group_name = site_group_map[site_name]
            
            if not target_group_name and site_name and site_name in zabbix_groups:
                target_group_name = site_name
                
            if not target_group_name:
                target_group_name = get_default_juniper_group_name(dev_type)

            group_id = zabbix_groups.get(target_group_name)
            if not group_id:
                return {
                    "status": "failed",
                    "host": serial,
                    "name": name,
                    "reason": f"Target group '{target_group_name}' not found in Zabbix"
                }

            # Templates
            templates = []
            target_tmpl_name = None
            if dev_type == "ap":
                target_tmpl_name = "Env_Juniper_Wireless"
            elif dev_type == "switch":
                target_tmpl_name = "Env_Juniper_Switch"
            
            if target_tmpl_name and target_tmpl_name in template_cache:
                templates.append({"templateid": template_cache[target_tmpl_name]})
            
            if "Env_Meraki_ICMP_Ping" in template_cache:
                templates.append({"templateid": template_cache["Env_Meraki_ICMP_Ping"]})

            # Macros
            macros = [
                {"macro": "{$SERIAL}", "value": serial or ""},
                {"macro": "{$TYPE}", "value": dev_type or ""},
                {"macro": "{$MAC}", "value": mac or ""},
                {"macro": "{$ID}", "value": dev_id or ""},
                {"macro": "{$SITE_ID}", "value": site_id or ""},
                {"macro": "{$JUNIPER.API.ORGID}", "value": org_id or ""},
                {"macro": "{$SNMP_COMMUNITY}", "value": "public"}
            ]
            
            if api_token:
                 macros.append({"macro": "{$JUNIPER.API.TOEKN}", "value": api_token})
            if juniper_api:
                 macros.append({"macro": "{$JUNIPER.API.URL}", "value": juniper_api})

            # Tags
            # User requested NOT to add any tags during creation.
            tags = []

            # Create Host
            # Handle visible name conflict by appending serial if needed
            # Since ZabbixService.create_host returns 'failed' on name conflict, 
            # we can try to catch it here or modify ZabbixService.
            # But ZabbixService is shared. So let's handle retry here.
            
            res = zabbix.create_host(serial, name, group_id, interfaces, templates=templates, macros=macros, tags=tags)
            
            # Retry logic for visible name conflict
            if res.get("status") == "failed" and "visible name" in str(res.get("error", "")).lower() and "already exists" in str(res.get("error", "")).lower():
                new_name = f"{name}_{serial}"
                logger.warning(f"Visible name conflict for {name} ({serial}). Retrying with {new_name}...")
                res = zabbix.create_host(serial, new_name, group_id, interfaces, templates=templates, macros=macros, tags=tags)
            
            res["group"] = target_group_name
            return res

        with concurrent.futures.ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
            future_to_dev = {executor.submit(_create_update_task, d): d for d in current_device_map.values()}
            for future in concurrent.futures.as_completed(future_to_dev):
                d_info = future_to_dev[future]
                s_serial = d_info.get("serial")
                try:
                    res = future.result()
                    if res["status"] == "created":
                        results["created"].append(res)
                    elif res["status"] in ["skipped", "updated_group", "updated_templates", "updated_all", "reactivated", "reactivated_and_updated"]:
                        results["skipped"].append(res)
                    else:
                        results["failed"].append(res)
                        logger.error(f"Failed to create/update Juniper host {s_serial}: {res}")
                except Exception as exc:
                    logger.error(f"Juniper task exception for {s_serial}: {exc}")
                    results["failed"].append({"host": s_serial, "error": str(exc)})

        logger.info(f"Juniper Sync complete. Created: {len(results['created'])}, Skipped/Updated: {len(results['skipped'])}")

        # 4. Update Cache & Backup
        new_synced_list = list(current_serials)
        set_json(ZABBIX_JUNIPER_SYNCED_HOSTS_KEY, new_synced_list)
        logger.info("Updated Juniper Redis sync cache.")

        # Backup to text file
        try:
            log_dir = settings.LOG_DIR
            backup_file = os.path.join(log_dir, "zabbix_juniper_synced_hosts.txt")
            with open(backup_file, "w", encoding="utf-8") as f:
                for s in new_synced_list:
                    f.write(f"{s}\n")
            logger.info(f"Backed up Juniper synced hosts list to {backup_file}")
        except Exception as e:
            logger.error(f"Failed to backup Juniper synced hosts to file: {e}")
        
        return JsonResponse(results, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

    except Exception as e:
        logger.exception("sync_juniper_to_zabbix failed")
        return JsonResponse({"error": str(e)}, status=500)
