"""
状态相关视图
"""
import logging
from django.http import JsonResponse

from meraki_Interface_forward.redis_utils import (
    get_json,
    CacheKey,
    get_or_set_json_with_lock,
)
from meraki_Interface_forward.services.meraki_service import (
    get_organization_status,
    get_organization_status_overview,
    get_organization_alert,
)
from meraki_Interface_forward.services.cache_service import (
    STATUS_CACHE_PREFIX,
    cache_status_by_serial,
)

logger = logging.getLogger("meraki_Interface_forward.views.status_views")


def get_device_status(request):
    """获取设备状态"""
    serial = request.GET.get('serial')
    product_type = request.GET.get('productType')

    try:
        if not serial or not product_type:
            return JsonResponse(
                {"message": "缺少查询参数 serial 或 productType"},
                status=400,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 1. 优先读单设备状态缓存（status:<serial>），O(1) 命中，TTL 65 秒
        status_cache_key = f"{STATUS_CACHE_PREFIX}{serial}"
        status_val = get_json(status_cache_key)
        if status_val is not None:
            # 缓存命中，直接返回
            return JsonResponse(
                {"status": status_val},
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 2. 单设备缓存未命中 → 从全量状态列表缓存中按需查找并回填
        res = get_json(CacheKey.DEVICES_AVAILABILITIES.value)
        if isinstance(res, list) and res:
            logger.debug(
                "单设备状态缓存未命中，尝试在全量状态缓存中查找并回填: serial=%s, 当前全量缓存条数=%d",
                serial,
                len(res),
            )
            for device in res:
                if (
                    isinstance(device, dict)
                    and device.get("serial") == serial
                    and device.get("productType") == product_type
                ):
                    status_val = device.get("status")
                    # 回填单设备状态缓存（只保存 status 值）
                    cache_status_by_serial([device], ttl=65)
                    return JsonResponse(
                        {"status": status_val},
                        safe=False,
                        json_dumps_params={'indent': 2, 'ensure_ascii': False},
                    )

        # 3. 全量缓存也未命中 → 使用带锁的全量刷新，批量保存所有设备状态
        def loader():
            data = get_organization_status()
            if data:
                # 一次性批量保存所有设备的状态到单设备缓存（只保存 status 值）
                cache_status_by_serial(data, ttl=65)
            return data

        res, _ = get_or_set_json_with_lock(
            CacheKey.DEVICES_AVAILABILITIES.value,
            loader=loader,
            ex=65,
        )

        if not res:
            return JsonResponse(
                {"error": "当前无法获取设备状态，请稍后重试"},
                status=503,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 4. 全量刷新后，再次尝试从单设备缓存读取
        status_val = get_json(status_cache_key)
        if status_val is not None:
            return JsonResponse(
                {"status": status_val},
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 5. 依然未找到设备状态
        return JsonResponse(
            { "error": "未找到指定设备的状态信息", "serial": serial, "productType": product_type},
            status = 404,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )
    except Exception as e:
        logger.exception("get_device_status 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


def get_device_status_overview(request):
    """获取设备状态概览"""
    try:
        # 先读缓存 + 带锁回源，避免缓存失效瞬间被大量请求打穿
        def loader():
            return get_organization_status_overview()

        res, _ = get_or_set_json_with_lock(
            CacheKey.DEVICES_STATUSES_OVERVIEW.value,
            loader=loader,
            ex=60,
        )
        if res:
            return JsonResponse(res, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})
        return JsonResponse([], safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})
    except Exception as e:
        logger.exception("get_device_status_overview 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


def get_device_alert(request):
    """获取设备告警"""
    try:
        serial = request.GET.get('serial')
        if not serial:
            return JsonResponse({"message": "缺少查询参数 serial"}, status=400)

        # 先读缓存，防止高并发下的缓存穿透
        def loader():
            return get_organization_alert()

        alert, _ = get_or_set_json_with_lock(
            CacheKey.ASSURANCE_ALERTS.value,
            loader=loader,
            ex=60,
        )
        result = []
        if alert:
            for res in alert:
                if res.get('serial') == serial:
                    result.append(res)

        return JsonResponse(result, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

    except Exception as e:
        logger.exception("get_device_alert 调用失败")
        return JsonResponse({"error": str(e)}, status=500)

