import os
import json
import time
import random
from typing import Any, Callable, Tuple, Optional

try:
    import redis
    _REDIS_AVAILABLE = True
except Exception:
    redis = None
    _REDIS_AVAILABLE = False

def _env_url():
    return os.getenv("REDIS_URL")

def _env_host():
    return os.getenv("REDIS_HOST", "127.0.0.1")

def _env_port():
    return int(os.getenv("REDIS_PORT", "6379"))

def _env_db():
    return int(os.getenv("REDIS_DB", "8"))

def _env_password():
    return os.getenv("REDIS_PASSWORD")

def _env_username():
    return os.getenv("REDIS_USERNAME")

def get_redis_client():
    if not _REDIS_AVAILABLE:
        return None
    host = os.getenv("REDIS_HOST")
    port = os.getenv("REDIS_PORT")
    db = os.getenv("REDIS_DB")
    username = _env_username()
    password = _env_password()
    if host or port or db or username or password:
        return redis.StrictRedis(
            host=_env_host(),
            port=_env_port(),
            db=_env_db(),
            username=username,
            password=password,
            decode_responses=True,
        )
    url = _env_url()
    if url:
        return redis.StrictRedis.from_url(url, decode_responses=True)
    return redis.StrictRedis(host=_env_host(), port=_env_port(), db=_env_db(), password=password, username=username, decode_responses=True)

def redis_ping():
    client = get_redis_client()
    if not client:
        return {"connected": False, "error": "redis library not available"}
    try:
        ok = client.ping()
        if ok:
            return {"connected": True}
        return {"connected": False}
    except Exception as e:
        return {"connected": False, "error": str(e)}

def set(key, value, ex=None):
    c = get_redis_client()
    if not c:
        return False
    return c.set(key, value, ex=ex)

def get(key):
    c = get_redis_client()
    if not c:
        return None
    return c.get(key)

def delete(key):
    c = get_redis_client()
    if not c:
        return 0
    return c.delete(key)

def exists(key):
    c = get_redis_client()
    if not c:
        return 0
    return c.exists(key)

def expire(key, seconds):
    c = get_redis_client()
    if not c:
        return False
    return c.expire(key, seconds)

def hset(name, key, value):
    c = get_redis_client()
    if not c:
        return 0
    return c.hset(name, key, value)

def hget(name, key):
    c = get_redis_client()
    if not c:
        return None
    return c.hget(name, key)

def set_json(key, obj, ex=None):
    return set(key, json.dumps(obj, ensure_ascii=False), ex=ex)

def get_json(key):
    val = get(key)
    if not val:
        return None
    try:
        return json.loads(val)
    except Exception:
        return None

def ttl(key):
    """Return remaining TTL seconds for key, or None if redis unavailable."""
    c = get_redis_client()
    if not c:
        return None
    try:
        return c.ttl(key)
    except Exception:
        return None

def rate_limit_allow(name: str, limit: int = 10, window_seconds: int = 1):
    """
    Simple sliding-window rate limiter using Redis INCR + EXPIRE.
    Returns (allowed: bool, remaining: int).

    - name: unique key for the rate limit bucket (e.g., rl:switch_ports:<serial>)
    - limit: max number of requests allowed within window
    - window_seconds: window duration in seconds
    """
    c = get_redis_client()
    # If Redis not available, allow request (best-effort fallback)
    if not c:
        return True, limit
    try:
        count = c.incr(name)
        if count == 1:
            # first hit in window, set expire
            c.expire(name, window_seconds)
        allowed = count <= limit
        remaining = max(limit - count, 0)
        return allowed, remaining
    except Exception:
        # On redis error, allow to avoid hard failure
        return True, limit


def scan_delete(pattern: str, batch_size: int = 500) -> int:
    """
    按模式批量删除 Redis key，返回删除的数量。
    - pattern: 如 'device:*', 'uplink:*', 'lock:*' 等
    - batch_size: 每次 scan 的数量
    仅用于运维类清理操作，避免 KEYS 带来的阻塞风险。
    """
    c = get_redis_client()
    if not c:
        return 0
    deleted = 0
    try:
        cursor = 0
        while True:
            cursor, keys = c.scan(cursor=cursor, match=pattern, count=batch_size)
            if keys:
                deleted += c.delete(*keys)
            if cursor == 0:
                break
    except Exception:
        # 清理失败时静默返回当前已删数量，避免影响业务
        return deleted
    return deleted


def _lock_key_for(cache_key: str) -> str:
    return f"lock:{cache_key}"


def acquire_lock(name: str, ttl_seconds: int = 10) -> bool:
    """
    获取一个简单的分布式锁，使用 Redis SET NX + EX。
    - name: 锁 key
    - ttl_seconds: 锁失效时间，避免死锁
    """
    c = get_redis_client()
    if not c:
        # 如果 Redis 不可用，则视为“拿不到锁”，避免逻辑过于复杂
        return False
    try:
        # 使用随机值可以在需要时做更安全的解锁，这里简单处理：只要 key 存在就认为有锁
        ok = c.set(name, str(time.time()), nx=True, ex=ttl_seconds)
        return bool(ok)
    except Exception:
        return False


def release_lock(name: str) -> None:
    """释放锁，错误时静默失败。"""
    c = get_redis_client()
    if not c:
        return
    try:
        c.delete(name)
    except Exception:
        return


def get_or_set_json_with_lock(
    cache_key: str,
    loader: Callable[[], Any],
    ex: int,
    lock_ttl: int = 10,
    wait_timeout: float = 5.0,
    wait_interval: float = 0.2,
) -> Tuple[Optional[Any], bool]:
    """
    通用的“带锁缓存读取”，防止高并发下的缓存击穿/穿透。

    返回 (value, loaded_from_source):
    - value: 最终得到的值（可能为 None）
    - loaded_from_source: 是否由本次调用触发 loader() 从源头加载

    行为：
    1. 先读缓存，命中直接返回。
    2. 未命中时，尝试获取分布式锁：
       - 成功：调用 loader()，将结果写回缓存并释放锁。
       - 失败：在 wait_timeout 内循环等待，期间轮询缓存，直到有值或超时。
    3. 等待超时仍未获取到值时，返回 (None, False)。上层可决定返回错误或空结构。
    """
    # Step 1: 先查缓存
    val = get_json(cache_key)
    if val is not None:
        return val, False

    # Step 2: 竞争锁
    lock_key = _lock_key_for(cache_key)
    if acquire_lock(lock_key, ttl_seconds=lock_ttl):
        try:
            loaded = loader()
            if loaded is not None:
                set_json(cache_key, loaded, ex=ex)
            return loaded, True
        finally:
            release_lock(lock_key)

    # Step 3: 未拿到锁 → 轮询等待其他进程/线程填充缓存，避免缓存穿透
    deadline = time.time() + wait_timeout
    while time.time() < deadline:
        time.sleep(wait_interval + random.uniform(0, wait_interval / 2))
        val = get_json(cache_key)
        if val is not None:
            return val, False

    # 等待超时，作为降级处理，交给上层决定怎么返回
    return None, False


from enum import Enum
class CacheKey(Enum):
    ORGANIZATIONS = "organizations"
    NETWORKS = "networks"
    DEVICES = "meraki_devices"
    DEVICES_AVAILABILITIES = "devices_availabilities"
    DEVICES_STATUSES_OVERVIEW = "devices_statuses_overview"
    DEVICES_STATUSES = "devices_statuses"
    DEVICES_UPLINKS_BY_DEVICE = "devices_uplinks_by_device"
    WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE = "wireless_channel_utilization_by_device"
    ASSURANCE_ALERTS = "assurance_alerts"