"""
缓存服务层：封装所有缓存相关操作
"""
import logging

from meraki_Interface_forward.redis_utils import (
    set_json,
    CacheKey,
)

logger = logging.getLogger("meraki_Interface_forward.services.cache_service")

# 单设备缓存前缀
DEVICE_CACHE_PREFIX = "device:"
UPLINK_CACHE_PREFIX = "uplink:"
STATUS_CACHE_PREFIX = "status:"
CHANNEL_UTILIZATION_CACHE_PREFIX = "channel_utilization:"


def cache_devices_by_serial(devices, ttl: int = 60 * 60 * 12) -> None:
    """
    将 Meraki 设备列表按 serial 维度拆分，逐个写入 Redis。
    - key: device:<serial>
    - value: 单个设备 JSON
    """
    if not isinstance(devices, list):
        return
    for dev in devices:
        if not isinstance(dev, dict):
            continue
        serial = dev.get("serial")
        if not serial:
            continue
        try:
            set_json(f"{DEVICE_CACHE_PREFIX}{serial}", dev, ex=ttl)
        except Exception:
            continue


def cache_uplinks_by_serial(uplinks, ttl: int = 60 * 60 * 6) -> None:
    """将组织级 uplink 列表按 serial 拆分缓存"""
    if not isinstance(uplinks, list):
        return
    for item in uplinks:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        try:
            set_json(f"{UPLINK_CACHE_PREFIX}{serial}", item, ex=ttl)
        except Exception:
            continue


def cache_status_by_serial(status_list, ttl: int = 65) -> None:
    """
    将组织级设备状态列表按 serial 拆分缓存，只保存 status 字段值
    - key: status:<serial>
    - value: 设备的 status 字段值（字符串，如 "online", "offline" 等）
    """
    if not isinstance(status_list, list):
        return
    for item in status_list:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        status_val = item.get("status")
        try:
            set_json(f"{STATUS_CACHE_PREFIX}{serial}", status_val, ex=ttl)
        except Exception:
            continue


def cache_channel_utilization_by_serial(utilization_list, ttl: int = 300) -> None:
    """将组织级信道利用率列表按 serial 拆分缓存"""
    if not isinstance(utilization_list, list):
        return
    for item in utilization_list:
        if not isinstance(item, dict):
            continue
        serial = item.get("serial")
        if not serial:
            continue
        try:
            set_json(f"{CHANNEL_UTILIZATION_CACHE_PREFIX}{serial}", item, ex=ttl)
        except Exception:
            continue

