"""
Meraki API 服务层：封装所有 Meraki API 调用
"""
import os
import meraki
import logging

from meraki_Interface_forward.redis_utils import CacheKey, set_json

logger = logging.getLogger("meraki_Interface_forward.services.meraki_service")

# Meraki API 配置
API_KEY = os.getenv("MERAKI_API_KEY")
ORGANIZATION_ID = os.getenv("ORGANIZATION_ID")
MERAKI_API_BASE = os.getenv("MERAKI_API_BASE")

dashboard = meraki.DashboardAPI(
    api_key=API_KEY,
    base_url=MERAKI_API_BASE,
    output_log=True,
    print_console=True,
    suppress_logging=True,
    wait_on_rate_limit=True,
    nginx_429_retry_wait_time=5,
    maximum_retries=3,
)


def get_organization_devices():
    """获取组织所有设备"""
    return dashboard.organizations.getOrganizationDevices(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all'
    )


def get_organization_networks():
    """获取组织所有网络"""
    try:
        networks = dashboard.organizations.getOrganizationNetworks(
            ORGANIZATION_ID,
            perPage=1000,
            total_pages='all'
        )
        if len(networks):
            set_json(CacheKey.NETWORKS.value, networks, ex=60*60*6)
            return networks
    except Exception as e:
        msg = str(e)
        if "Expecting value: line 1 column 1" in msg:
            return []
        logger.error(f"获取组织网络失败: {e}")
        return None
    return None


def get_organization_status():
    """获取组织所有设备状态"""
    return dashboard.organizations.getOrganizationDevicesAvailabilities(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all'
    )


def get_organization_status_overview():
    """获取组织所有设备状态概览"""
    return dashboard.organizations.getOrganizationDevicesStatusesOverview(ORGANIZATION_ID)


def get_organization_alert():
    """获取组织所有设备告警"""
    alerts = dashboard.organizations.getOrganizationAssuranceAlerts(
        ORGANIZATION_ID, total_pages="all"
    )

    processed = []
    if alerts:
        for item in alerts or []:
            scope = item.get("scope", {})
            devices = scope.get("devices", [])

            # 有多个设备 → 每个设备都生成一条数据
            if devices:
                for dev in devices:
                    processed.append({
                        "id": item.get("id"),
                        "categoryType": item.get("categoryType"),
                        "networkId": item.get("network", {}).get("id"),
                        "networkName": item.get("network", {}).get("name"),
                        "type": item.get("type"),
                        "title": item.get("title"),
                        "severity": item.get("severity"),
                        "deviceType": item.get("deviceType"),
                        "startedAt": item.get("startedAt"),
                        "resolvedAt": item.get("resolvedAt"),
                        "dismissedAt": item.get("dismissedAt"),
                        # 多设备的关键字段
                        "serial": dev.get("serial"),
                        "mac": dev.get("mac"),
                        "deviceName": dev.get("name"),
                        "port": dev.get("portIdentifier"),
                        "scope": scope
                    })
    return processed


def get_organization_uplinks():
    """获取组织所有设备 uplink"""
    return dashboard.organizations.getOrganizationDevicesUplinksAddressesByDevice(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all',
    )


def get_organization_channel_utilization():
    """获取组织所有设备信道利用率"""
    return dashboard.wireless.getOrganizationWirelessDevicesChannelUtilizationByDevice(
        ORGANIZATION_ID,
        perPage=1000,
        total_pages='all',
        timespan=3600
    )


def get_device_switch_ports_status(serial):
    """获取单台交换机端口状态"""
    return dashboard.switch.getDeviceSwitchPortsStatuses(
        serial,
        timespan=3600,
    )

