"""
设备相关视图
"""
import logging
import json
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt

from meraki_Interface_forward.redis_utils import (
    get_json,
    set_json,
    CacheKey,
    get_or_set_json_with_lock,
)
from meraki_Interface_forward.services.meraki_service import (
    get_organization_devices,
    get_organization_uplinks,
    get_organization_channel_utilization,
    get_device_switch_ports_status as fetch_device_switch_ports_status,
)
from meraki_Interface_forward.services.cache_service import (
    DEVICE_CACHE_PREFIX,
    UPLINK_CACHE_PREFIX,
    CHANNEL_UTILIZATION_CACHE_PREFIX,
    cache_devices_by_serial,
    cache_uplinks_by_serial,
    cache_channel_utilization_by_serial,
)
from meraki_Interface_forward.redis_utils import rate_limit_allow

logger = logging.getLogger("meraki_Interface_forward.views.device_views")


def discovery_host_prototype(request):
    """设备发现接口"""
    try:
        # 先读 Redis
        devices = get_json(CacheKey.DEVICES.value)
        if devices:
            return JsonResponse(devices, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

        # 缓存未命中，回退 Meraki API，并顺便按 serial 建立索引
        devices = get_organization_devices()
        if devices:
            set_json(CacheKey.DEVICES.value, devices, ex=43200)
            cache_devices_by_serial(devices, ttl=43200)
            return JsonResponse(devices, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

        return JsonResponse([], safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})
    except Exception as e:
        logger.exception("discovery_host_prototype 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


@csrf_exempt
def discovery_host_prototype_filter(request):
    """
    POST 接口：支持按网络名称过滤设备（过滤命中的网络，将设备剔除）
    body 示例：
    {
       "filter_networks": ["网络1", "网络2"]
    }
    """
    if request.method != "POST":
        return JsonResponse({"message": "Method Not Allowed"}, status=405)

    try:
        body = {}
        if request.body:
            body = json.loads(request.body.decode("utf-8"))
        filter_networks = body.get("filter_networks") or []
        filter_names = set(filter_networks) if isinstance(filter_networks, list) else set()

        # 获取网络列表（缓存优先）
        networks = get_json(CacheKey.NETWORKS.value) or []
        if isinstance(networks, str):
            try:
                networks = json.loads(networks)
            except Exception:
                networks = []

        # 网络名 -> id 映射，用于过滤
        exclude_network_ids = set()
        if filter_names and isinstance(networks, list):
            for net in networks:
                if isinstance(net, dict) and net.get("name") in filter_names:
                    nid = net.get("id")
                    if nid:
                        exclude_network_ids.add(nid)

        # 获取设备（缓存优先，未命中则回源）
        devices = get_json(CacheKey.DEVICES.value)
        if not devices:
            devices = get_organization_devices()
            if devices:
                set_json(CacheKey.DEVICES.value, devices, ex=43200)
                cache_devices_by_serial(devices, ttl=43200)

        if not devices:
            return JsonResponse([], safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

        # 过滤：networkId 命中需要排除的网络则丢弃
        if exclude_network_ids:
            filtered = []
            for dev in devices:
                if not isinstance(dev, dict):
                    continue
                nid = dev.get("networkId")
                if nid in exclude_network_ids:
                    continue
                filtered.append(dev)
            devices = filtered

        return JsonResponse(devices, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

    except Exception as e:
        logger.exception("discovery_host_prototype_filter 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


def get_device_by_serial_or_product_type(request):
    """
    根据 serial + productType 查询单个设备信息。
    优化点：
    - 使用按 serial 维度的单设备缓存（device:<serial>），O(1) 查询
    - 若单设备缓存未命中，则通过带锁的全量缓存一次性拉取并拆分写入
    """
    if request.method != 'GET':
        return JsonResponse({"message": "Method Not Allowed"}, status=405)

    try:
        serial = request.GET.get('serial')
        product_type = request.GET.get('productType')

        if not serial or not product_type:
            return JsonResponse({"message": "缺少查询参数 serial 或 productType"}, status=400)

        logger.info("get_device_by_serial_or_product_type 请求: serial=%s, productType=%s", serial, product_type)
        network_map = {}
        # 1. 获取 Network 映射
        networks = get_json(CacheKey.NETWORKS.value)
        if networks:
            network_map = {net["id"]: net["name"] for net in networks}

        # 2. 优先通过单设备缓存（device:<serial>）查询，O(1) 时间复杂度
        cache_key = f"{DEVICE_CACHE_PREFIX}{serial}"
        device = get_json(cache_key)
        if device and device.get("productType") == product_type:
            net_id = device.get("networkId")
            if network_map and net_id:
                device["networkName"] = network_map.get(net_id)
            logger.debug("单设备缓存命中: serial=%s", serial)
            return JsonResponse(
                {"device": device},
                safe=False,
                json_dumps_params={"indent": 2, "ensure_ascii": False},
            )

        # 3. 单设备缓存未命中 → 尝试从全量列表缓存中按需回填一条
        meraki_devices = get_json(CacheKey.DEVICES.value)
        if isinstance(meraki_devices, list) and meraki_devices:
            logger.debug(
                "单设备缓存未命中，尝试在全量缓存中查找并回填: serial=%s, 当前全量缓存条数=%d",
                serial,
                len(meraki_devices),
            )
            for dev in meraki_devices:
                if (
                    isinstance(dev, dict)
                    and dev.get("serial") == serial
                    and dev.get("productType") == product_type
                ):
                    # 命中后，顺便写回单设备缓存
                    cache_devices_by_serial([dev], ttl=43200)
                    net_id = dev.get("networkId")
                    if network_map and net_id:
                        dev["networkName"] = network_map.get(net_id)
                    logger.info("在全量缓存中命中设备并已回填: serial=%s", serial)
                    return JsonResponse(
                        {"device": dev},
                        safe=False,
                        json_dumps_params={"indent": 2, "ensure_ascii": False},
                    )

        # 4. 单设备缓存与全量缓存都未命中 → 使用带锁的全量刷新
        def loader():
            logger.info("单设备缓存未命中，准备触发全量设备加载: serial=%s", serial)
            devs = get_organization_devices()
            if devs:
                set_json(CacheKey.DEVICES.value, devs, ex=43200)
                cache_devices_by_serial(devs, ttl=43200)
            return devs

        _, _ = get_or_set_json_with_lock(
            CacheKey.DEVICES.value,
            loader=loader,
            ex=43200,
            lock_ttl=30,
            wait_timeout=5.0,
            wait_interval=0.2,
        )

        # 5. 再次尝试从单设备缓存读取
        device = get_json(cache_key)
        if device and device.get("productType") == product_type:
            net_id = device.get("networkId")
            if network_map and net_id:
                device["networkName"] = network_map.get(net_id)
            logger.info("通过全量加载后在缓存中找到设备: serial=%s", serial)
            return JsonResponse(
                {"device": device},
                safe=False,
                json_dumps_params={"indent": 2, "ensure_ascii": False},
            )

        # 6. 依然未找到设备
        logger.warning("未找到匹配的设备: serial=%s, productType=%s", serial, product_type)
        return JsonResponse(
            {"error": "未找到匹配的设备"},
            status=404,
            safe=False,
            json_dumps_params={"indent": 2, "ensure_ascii": False},
        )

    except Exception as e:
        logger.exception("get_device_by_serial_or_product_type 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


def get_device_uplink(request):
    """获取设备 uplink 信息"""
    serial = request.GET.get('serial')
    product_type = request.GET.get('productType')
    try:
        if not serial or not product_type:
            return JsonResponse({"message": "缺少查询参数 serial 或 productType"}, status=400)

        # 先读单设备 uplink 缓存，O(1) 命中
        uplink_cache_key = f"{UPLINK_CACHE_PREFIX}{serial}"
        uplink = get_json(uplink_cache_key)
        if uplink and uplink.get("productType") == product_type:
            return JsonResponse(
                uplink,
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 单设备缓存未命中 → 读取组织级 uplink 列表（带锁防击穿）
        def loader():
            data = get_organization_uplinks()
            if data:
                cache_uplinks_by_serial(data, ttl=43200)
            return data

        res, _ = get_or_set_json_with_lock(
            CacheKey.DEVICES_UPLINKS_BY_DEVICE.value,
            loader=loader,
            ex=43200,
        )

        if not res:
            return JsonResponse(
                {},
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 在组织级列表中按需查找本设备，并回填单设备缓存
        for device in res:
            if (
                isinstance(device, dict)
                and device.get('serial') == serial
                and device.get('productType') == product_type
            ):
                cache_uplinks_by_serial([device], ttl=43200)
                return JsonResponse(
                    device,
                    safe=False,
                    json_dumps_params={'indent': 2, 'ensure_ascii': False},
                )

        # 未找到匹配 uplink
        return JsonResponse(
            {"error": "未找到设备 uplink 信息", "serial": serial, "productType": product_type},
            status=404,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )
    except Exception as e:
        logger.exception("get_device_uplink 调用失败")
        return JsonResponse({"error": str(e)}, status=500)


def get_device_ap_channelUtilization(request):
    """获取设备 AP 信道利用率"""
    try:
        serial = request.GET.get('serial')
        if not serial:
            return JsonResponse({"message": "缺少查询参数 serial"}, status=400)

        # 1. 优先读单设备信道利用率缓存
        channel_cache_key = f"{CHANNEL_UTILIZATION_CACHE_PREFIX}{serial}"
        channel_data = get_json(channel_cache_key)
        if channel_data:
            return JsonResponse(
                channel_data,
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 2. 单设备缓存未命中 → 从全量信道利用率列表缓存中按需查找并回填
        res = get_json(CacheKey.WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE.value)
        if isinstance(res, list) and res:
            logger.debug(
                "单设备信道利用率缓存未命中，尝试在全量缓存中查找并回填: serial=%s, 当前全量缓存条数=%d",
                serial,
                len(res),
            )
            for device in res:
                if isinstance(device, dict) and device.get('serial') == serial:
                    cache_channel_utilization_by_serial([device], ttl=300)
                    return JsonResponse(
                        device,
                        safe=False,
                        json_dumps_params={'indent': 2, 'ensure_ascii': False},
                    )

        # 3. 全量缓存也未命中 → 使用带锁的全量刷新
        def loader():
            data = get_organization_channel_utilization()
            if data:
                cache_channel_utilization_by_serial(data, ttl=300)
            return data

        res, _ = get_or_set_json_with_lock(
            CacheKey.WIRELESS_CHANNEL_UTILIZATION_BY_DEVICE.value,
            loader=loader,
            ex=43200,
        )

        if not res:
            # 冷启动/源数据为空，返回 503 提示稍后重试
            return JsonResponse(
                {"error": "信道利用率数据尚未准备好，请稍后重试"},
                status=503,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 4. 全量刷新后，再次尝试从单设备缓存读取
        channel_data = get_json(channel_cache_key)
        if channel_data:
            return JsonResponse(
                channel_data,
                safe=False,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )

        # 5. 依然未找到设备信道利用率数据
        return JsonResponse(
            {"error": "未找到指定设备的信道利用率数据", "serial": serial},
            status=404,
            json_dumps_params={'indent': 2, 'ensure_ascii': False},
        )

    except Exception as e:
        msg = str(e)
        if "Expecting value: line 1 column 1" in msg:
            return JsonResponse(
                {"error": "Meraki 返回空响应或非 JSON，可能为 204/HTML 页面。请检查 base_url、组织访问权限，或稍后重试。"},
                status=502,
                json_dumps_params={'indent': 2, 'ensure_ascii': False},
            )
        logger.exception("get_device_ap_channelUtilization 调用失败")
        return JsonResponse({"error": msg}, status=500)


def get_device_switch_ports_status(request):
    """
    获取单台交换机端口状态。
    优化点：
    - 优先读缓存，避免高并发下直接打穿 Meraki API
    - 使用 Redis 分布式锁避免缓存击穿
    - 仍保留每设备限流，保护后端和 Meraki
    """
    try:
        serial = request.GET.get('serial')
        if not serial:
            return JsonResponse({"message": "缺少查询参数 serial"}, status=400)

        cache_key = f"switch_ports_status:{serial}"

        def loader():
            # 每设备接口限流：8次/秒
            allowed, remaining = rate_limit_allow(f"rl:switch_ports:{serial}", limit=8, window_seconds=1)
            if not allowed:
                logger.warning(
                    "switch_ports_status 限流触发: serial=%s limit=%d remaining=%d",
                    serial, 8, remaining
                )
                return None

            # 真正调用 Meraki API
            res = fetch_device_switch_ports_status(serial)
            # Meraki 可能返回空列表，统一返回 list 以便缓存
            return res if res is not None else []

        # 使用带锁缓存读取，TTL 60 秒
        data, loaded_from_source = get_or_set_json_with_lock(
            cache_key,
            loader=loader,
            ex=60,
            lock_ttl=10,
            wait_timeout=5.0,
            wait_interval=0.2,
        )

        if data is None:
            return JsonResponse(
                {"error": "当前请求过多或后端暂时不可用，请稍后重试"},
                status=503,
            )

        # data 至少是 list（可能为空），保证接口幂等
        return JsonResponse(data, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

    except Exception as e:
        logger.exception("get_device_switch_ports_status 调用失败")
        return JsonResponse({"error": str(e)}, status=500)

