import logging
import requests
import json
import threading
from django.conf import settings

logger = logging.getLogger("meraki_Interface_forward.services.zabbix_service")

class ZabbixService:
    def __init__(self):
        self.url = settings.ZABBIX_API_URL
        self.user = settings.ZABBIX_USER
        self.password = settings.ZABBIX_PASSWORD
        self.auth_token = None
        self.request_id = 1
        self._lock = threading.RLock()
        
        self.session = requests.Session()
        adapter = requests.adapters.HTTPAdapter(pool_connections=20, pool_maxsize=20)
        self.session.mount('http://', adapter)
        self.session.mount('https://', adapter)

    def _request(self, method, params=None):
        if not self.url:
             raise ValueError("Zabbix URL not configured")
        
        with self._lock:
            auth = self.auth_token
            current_id = self.request_id
            self.request_id += 1

        # user.login 不需要 auth 字段
        if method == "user.login":
            auth = None

        payload = {
            "jsonrpc": "2.0",
            "method": method,
            "params": params or {},
            "id": current_id,
            "auth": auth
        }
        
        headers = {'Content-Type': 'application/json-rpc'}
        try:
            response = self.session.post(self.url, json=payload, headers=headers, timeout=30)
            response.raise_for_status()
            result = response.json()
            if 'error' in result:
                raise Exception(f"Zabbix API Error: {result['error']}")
            return result.get('result')
        except Exception as e:
            logger.error(f"Zabbix request failed: {method}, error: {e}")
            raise

    def login(self):
        if self.auth_token:
            return
        
        with self._lock:
            if self.auth_token:
                return
            
            try:
                # Zabbix API 6.0+ uses 'user' (not 'username') but some older libs/docs might confuse this.
                # The error "Invalid parameter "/": unexpected parameter "user"." suggests 
                # the server might be expecting 'username' (older versions < 5.4) OR there is a strict validation issue.
                # However, standard 6.0/7.0 is 'user'. 
                # If the error persists with 'user', try 'username' as fallback or check API version.
                # Let's try to detect if we should use 'username' based on error, or just try 'username' if 'user' fails.
                
                try:
                    self.auth_token = self._request("user.login", {
                        "user": self.user,
                        "password": self.password
                    })
                except Exception as e:
                    # If "unexpected parameter "user"" occurs, retry with "username"
                    if "unexpected parameter" in str(e) and "user" in str(e):
                        logger.warning("Zabbix login with 'user' failed, retrying with 'username'...")
                        self.auth_token = self._request("user.login", {
                            "username": self.user,
                            "password": self.password
                        })
                    else:
                        raise e

                logger.info("Zabbix login successful")
            except Exception as e:
                logger.error(f"Zabbix login failed: {e}")
                raise

    def get_hostgroups(self):
        """
        Get all host groups
        Return: {group_name: group_id}
        """
        self.login()
        groups = self._request("hostgroup.get", {
            "output": ["groupid", "name"],
        })
        
        group_map = {}
        if isinstance(groups, list):
            for group in groups:
                group_map[group['name']] = group['groupid']
        return group_map

    def get_template_id_by_name(self, template_name):
        """
        Query template ID by name
        """
        self.login()
        templates = self._request("template.get", {
            "output": ["templateid", "name"],
            "filter": {
                "host": [template_name] # In Zabbix API, template name is also queried via 'host' field or 'name' depending on version/context, usually 'host' works for technical name
            }
        })
        if templates and isinstance(templates, list):
            return templates[0].get("templateid")
        return None

    def disable_host(self, host_name):
        """
        Disable host by technical name (host field in Zabbix)
        """
        self.login()
        hosts = self._request("host.get", {
            "filter": {"host": [host_name]},
            "output": ["hostid"]
        })
        if hosts and isinstance(hosts, list):
            host_id = hosts[0]['hostid']
            try:
                # status: 1 = unmonitored (disabled)
                result = self._request("host.update", {"hostid": host_id, "status": 1})
                logger.info(f"Host {host_name} disabled: {result}")
                return {"status": "disabled", "host": host_name}
            except Exception as e:
                logger.error(f"Failed to disable host {host_name}: {e}")
                return {"status": "failed_disable", "host": host_name, "error": str(e)}
        return {"status": "not_found", "host": host_name}

    def update_host_group(self, host_id, new_group_id, append=False):
        """
        Update host group.
        :param host_id: Zabbix Host ID
        :param new_group_id: Target Group ID
        :param append: True to append group, False to replace ALL groups with new one
        """
        self.login()
        
        groups_payload = [{"groupid": new_group_id}]
        
        if append:
            # Get current groups
            current_host = self._request("host.get", {
                "hostids": host_id,
                "selectGroups": ["groupid"]
            })
            
            if current_host and isinstance(current_host, list):
                existing_groups = current_host[0].get("groups", [])
                current_group_ids = {g["groupid"] for g in existing_groups}
                
                if str(new_group_id) in current_group_ids:
                    return {"status": "skipped", "reason": "Already in group"}
                
                # Merge
                for g in existing_groups:
                    groups_payload.append({"groupid": g["groupid"]})

        try:
            # host.update replaces the 'groups' list
            self._request("host.update", {
                "hostid": host_id,
                "groups": groups_payload
            })
            return {"status": "updated", "hostid": host_id, "groups": groups_payload}
        except Exception as e:
            logger.error(f"Failed to update host {host_id} group: {e}")
            return {"status": "failed", "error": str(e)}

    def update_host_details(self, host_name, description=None, tags=None, status=None):
        """
        更新主机的描述、标签和状态
        :param host_name: 主机技术名称 (host technical name)
        :param description: 新的描述文本 (str)
        :param tags: 标签列表 (list of dict), e.g. [{"tag": "Role", "value": "Switch"}]
                     注意: Zabbix API 的 host.update 会覆盖原有 tags，如需追加请先查询。
        :param status: 状态 (0: 启用, 1: 禁用)
        """
        self.login()
        
        # 1. 获取 Host ID
        hosts = self._request("host.get", {
            "filter": {"host": [host_name]},
            "output": ["hostid"]
        })
        
        if not hosts or not isinstance(hosts, list):
            return {"status": "not_found", "host": host_name}
            
        host_id = hosts[0]['hostid']
        
        # 2. 构建更新参数
        params = {"hostid": host_id}
        
        if description is not None:
            params["description"] = description
            
        if tags is not None:
            params["tags"] = tags
            
        if status is not None:
            params["status"] = int(status) # 确保是整数
            
        # 如果没有要更新的字段，直接返回
        if len(params) <= 1:
            return {"status": "skipped", "reason": "No fields to update", "host": host_name}

        try:
            # 3. 调用 host.update
            result = self._request("host.update", params)
            logger.info(f"Host {host_name} updated details: {params.keys()}")
            return {"status": "updated", "host": host_name, "hostid": result['hostids'][0]}
        except Exception as e:
            logger.error(f"Failed to update host {host_name}: {e}")
            return {"status": "failed", "host": host_name, "error": str(e)}

    def update_host_templates(self, host_id, new_template_ids, append=True):
        """
        Update host templates.
        :param host_id: Zabbix Host ID
        :param new_template_ids: List of new template IDs (list of strings or ints)
        :param append: True to append templates (merge), False to replace ALL templates
        """
        self.login()
        
        templates_payload = [{"templateid": str(tid)} for tid in new_template_ids]
        
        if append:
            # Get current templates
            current_host = self._request("host.get", {
                "hostids": host_id,
                "selectParentTemplates": ["templateid"]
            })
            
            if current_host and isinstance(current_host, list):
                existing_templates = current_host[0].get("parentTemplates", [])
                current_template_ids = {t["templateid"] for t in existing_templates}
                
                # Filter out templates that are already linked
                to_add = []
                for tid in new_template_ids:
                    if str(tid) not in current_template_ids:
                        to_add.append({"templateid": str(tid)})
                
                if not to_add:
                    return {"status": "skipped", "reason": "All templates already linked"}
                
                # Merge existing with new ones
                templates_payload = [{"templateid": t["templateid"]} for t in existing_templates]
                templates_payload.extend(to_add)

        try:
            # host.update replaces the 'templates' list
            self._request("host.update", {
                "hostid": host_id,
                "templates": templates_payload
            })
            return {"status": "updated", "hostid": host_id, "templates": templates_payload}
        except Exception as e:
            logger.error(f"Failed to update host {host_id} templates: {e}")
            return {"status": "failed", "error": str(e)}

    def update_host_inventory_mode(self, host_id, mode=1):
        """
        Update host inventory mode.
        :param host_id: Zabbix Host ID
        :param mode: 1=Automatic, 0=Manual, -1=Disabled
        """
        self.login()
        try:
            self._request("host.update", {
                "hostid": host_id,
                "inventory_mode": int(mode)
            })
            return {"status": "updated", "hostid": host_id, "inventory_mode": mode}
        except Exception as e:
            logger.error(f"Failed to update host {host_id} inventory mode: {e}")
            return {"status": "failed", "error": str(e)}

    def create_host(self, host_technical_name, visible_name, group_id, interfaces, templates=None, macros=None, tags=None):
        """
        Create a host in Zabbix with advanced options.
        If host exists, it will re-activate the host and update its group/templates.
        :param host_technical_name: Technical name (must be unique, e.g. serial)
        :param visible_name: Visible name (can be non-unique, e.g. device name)
        :param templates: List of template IDs, e.g. [{"templateid": "10001"}]
        :param macros: List of macros, e.g. [{"macro": "{$MY_MACRO}", "value": "val"}]
        :param tags: List of tags, e.g. [{"tag": "Role", "value": "Switch"}]
        """
        self.login()
        
        # Check if host exists (by technical name)
        existing = self._request("host.get", {
            "filter": {"host": [host_technical_name]},
            "output": ["hostid", "status"]
        })
        
        if existing:
            host_id = existing[0]['hostid']
            # current_status = existing[0]['status']
            status = "skipped"
            msg_parts = ["Already exists"]
            
            # 0. Check and Re-activate logic REMOVED based on user request.
            # If a host is disabled (status=1), we leave it as is.
            # We only proceed to update Group and Templates.

            # 1. Update Group (Replace mode)
            group_update_res = self.update_host_group(host_id, group_id, append=False)
            if group_update_res.get("status") == "updated":
                status = "updated_group"
                msg_parts.append("Group updated")
            elif group_update_res.get("status") == "failed":
                 msg_parts.append(f"Group update failed: {group_update_res.get('error')}")

            # 2. Update Templates (Append mode)
            if templates:
                template_ids = [t['templateid'] for t in templates if 'templateid' in t]
                if template_ids:
                    tmpl_update_res = self.update_host_templates(host_id, template_ids, append=True)
                    if tmpl_update_res.get("status") == "updated":
                        status = "updated_templates" if status == "skipped" else "updated_all"
                        msg_parts.append("Templates updated")
                    elif tmpl_update_res.get("status") == "failed":
                        msg_parts.append(f"Templates update failed: {tmpl_update_res.get('error')}")

            # 3. Update Inventory Mode (Ensure it is Automatic)
            inv_res = self.update_host_inventory_mode(host_id, mode=1)
            if inv_res.get("status") == "updated":
                 # Usually we don't need to log this unless it failed, or if we want verbose details
                 pass
            elif inv_res.get("status") == "failed":
                 msg_parts.append(f"Inventory update failed: {inv_res.get('error')}")

            msg = "; ".join(msg_parts)
            logger.info(f"Host {host_technical_name} processed: {msg}")
            return {"status": status, "host": host_technical_name, "name": visible_name, "reason": msg, "hostid": host_id}

        params = {
            "host": host_technical_name,
            "name": visible_name,
            "interfaces": interfaces,
            "groups": [{"groupid": group_id}],
            "templates": templates or [],
            "macros": macros or [],
            "tags": tags or [],
            "inventory_mode": 1  # 1 = Automatic inventory mode
        }
        
        # Clean empty lists
        if not params["templates"]: del params["templates"]
        if not params["macros"]: del params["macros"]
        if not params["tags"]: del params["tags"]
        
        try:
            result = self._request("host.create", params)
            logger.info(f"Host {host_technical_name} ({visible_name}) created: {result}")
            return {"status": "created", "host": host_technical_name, "name": visible_name, "hostid": result['hostids'][0]}
        except Exception as e:
            logger.error(f"Failed to create host {host_technical_name}: {e}")
            return {"status": "failed", "host": host_technical_name, "name": visible_name, "error": str(e)}
