import json
import logging
import os
import concurrent.futures
from django.conf import settings
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt
from meraki_Interface_forward.services.meraki_service import get_filtered_devices
from meraki_Interface_forward.services.zabbix_service import ZabbixService
from meraki_Interface_forward.redis_utils import get_json, set_json

logger = logging.getLogger("meraki_Interface_forward.views.zabbix_views")

# Redis Key for storing synced host serials (Set or List)
ZABBIX_SYNCED_HOSTS_KEY = "zabbix:synced_hosts"

# Max workers for concurrent Zabbix API calls
MAX_WORKERS = 20

DEFAULT_NETWORK_GROUP_MAP = {
    "东莞办公室": "东莞办公室",
    "丰宁二期-工厂": "丰宁工厂",
    "乌兰二期-工厂": "乌兰察布工厂",
    "乌兰察布工厂IT网络": "乌兰察布工厂",
    "乐亭-工厂": "乐亭工厂",
    "五河-工厂": "五河工厂",
    "包头主机工厂": "包头主机工厂",
    "包头储能工厂": "包头储能工厂",
    "包头合金办公室": "包头合金工厂",
    "包头合金工厂IT网络": "包头合金工厂",
    "北京-办公室": "北京办公室",

    "博荟广场C座5F": "上海博荟广场",
    "博荟广场C座6F": "上海博荟广场",
    "博荟广场C座15F": "上海博荟广场",
    "博荟广场C座22F": "上海博荟广场",

    "台前云中心": "台前工厂",
    "呼和浩特办公室": "呼和浩特办公室",
    "商都叶片工厂": "商都叶片工厂",
    "大连-庄河工厂": "庄河工厂",
    "天津-办公室": "天津办公室",
    "如东叶片-工厂": "如东叶片工厂",
    "山东-单县塔基工厂": "单县塔基工厂",
    "巴彦淖尔二期-工厂": "巴彦淖尔工厂",
    "广西南宁-工厂": "南宁工厂",
    # "广西合浦-辅房": "停产", 
    "广西桂平-工厂": "桂平工厂",
    "庆阳-工厂": "庆阳工厂",
    "新疆-吉木乃工厂": "吉木乃工厂",
    "新疆乌鲁木齐办公室": "乌鲁木齐办公室",

    "武威储能": "武威工厂",
    "武威叶片-工厂": "武威工厂",
    "武威叶片宿舍食堂": "武威工厂",

    "江阴RDC外仓": "江阴RDC外仓",

    "江阴一期堆场": "江阴一期工厂",
    "江阴一期篮球场": "江阴一期工厂",
    "江阴一期车间": "江阴一期工厂",
    "江阴一期辅房": "江阴一期工厂",

    "江阴国家级供应商外仓": "江阴二期工厂",
    "江阴宝湾国际物流园": "江阴二期工厂",
    "江阴渔光会馆C区": "江阴二期工厂",
    "江阴二期车间": "江阴二期工厂",
    "江阴二期辅房": "江阴二期工厂",
    "江阴-口罩厂": "江阴二期工厂",

    "江阴三期": "江阴三期工厂",
    "江阴四期-仓库": "江阴四期工厂",
    "江阴四期变压办公室": "江阴四期工厂",

    "江阴石庄仓库": "江阴齿轮箱厂",
    "江阴齿轮办公室1F": "江阴齿轮箱厂",
    "江阴传动链-工厂": "江阴齿轮箱厂",

    "江阴制氢办公室": "江阴制氢办公室",
    "江阴氢能星球工厂2#厂房": "江阴制氢办公室",

    "江阴-小湖工厂": "江阴小湖工厂",

    "沈阳工厂": "沈阳工厂",

    "沙尔沁-工厂": "沙尔沁工厂",
    "沙尔沁-辅房": "沙尔沁工厂",
    "沙尔沁储能": "沙尔沁储能",

    "河北沧州塔基工厂": "沧州塔基工厂",
    "海兴工厂": "海兴工厂",
    "海兴工厂二期": "海兴工厂",

    "淮安-盱眙工厂": "盱眙工厂",
    "濮阳工厂": "濮阳工厂",
    "白城-工厂": "白城工厂",

    "翁牛特旗-工厂": "翁牛特旗工厂",
    "翁牛特旗-辅房": "翁牛特旗工厂",

    "苍南轴承工厂": "苍南轴承工厂",
    "若羌工厂": "若羌工厂",
    "襄阳工厂": "襄阳工厂",

    "赤峰元宝山P1期": "赤峰制氢工厂",
    "赤峰元宝山制氢工厂": "赤峰制氢工厂",
    "赤峰元宝山氢能3#": "赤峰制氢工厂",

    "郎溪工厂": "郎溪工厂",
    "酒泉工厂": "酒泉工厂",

    "钦州-三期叶片工厂": "钦州工厂",
    "陕西榆林云中心": "榆林工厂",

    "高安-工厂": "高安工厂",
    "高安-辅房": "高安工厂",
    "魏县-工厂": "魏县工厂",

    # ===== 海外 ===== 
    "Bangalore Office": "印度办公室",
    "Bangalore Office 11F": "印度办公室",
    "Boston-Office": "波士顿办公室",
    "Boulder-Office": "博尔德办公室",
    "Brazil-Office": "巴西办公室",
    "Denmark Office": "丹麦办公室",
    "DuBai-Office": "迪拜办公室",
    "GIC-WorkShop": "GIC-WorkShop",
    "London-Office": "伦敦办公室",
    "Melbourne-Office": "墨尔本办公室",
    "Menlo Park": "Menlo Park",
    "Singapore Office": "新加坡办公室",
    "Spain-Office": "西班牙办公室"
}


def get_default_zabbix_group_name(product_type):
     if product_type == "wireless":
         return "思科设备/无线设备AP"
     if product_type == "switch":
         return "思科设备/交换机"
     return "思科设备/其他"


@csrf_exempt
def sync_meraki_to_zabbix(request):
    """
    同步 Meraki 设备到 Zabbix 主机
    Body: {
        "filter_networks": [...],
        "networkGroupMap": {...}
    }
    """
    if request.method != "POST":
        return JsonResponse({"message": "Method Not Allowed"}, status=405)

    try:
        logger.info("Starting Meraki to Zabbix synchronization task...")
        
        body = {}
        if request.body:
            body = json.loads(request.body.decode("utf-8"))
        
        filter_networks = body.get("filter_networks")
        custom_map = body.get("networkGroupMap") or {}
        meraki_url_base = body.get("merakiApi")
        
        disable_host_tag = body.get("disableHostTag")
        
        logger.info(f"Request parameters - Filter Networks: {len(filter_networks) if filter_networks else 0}, Custom Map Keys: {list(custom_map.keys())}, Meraki API: {meraki_url_base}")

        # Merge maps: custom overrides default
        network_group_map = DEFAULT_NETWORK_GROUP_MAP.copy()
        if isinstance(custom_map, dict):
            network_group_map.update(custom_map)

        # 1. Get Devices
        logger.info("Fetching Meraki devices with filters...")
        devices = get_filtered_devices(filter_networks)
        logger.info(f"Fetched {len(devices)} devices from Meraki service.")
        
        # Even if devices is empty, we might need to disable hosts, so we don't return early if devices is empty,
        # unless it was truly an error. get_filtered_devices returns [] on empty.
        current_device_map = {} # serial -> device
        if devices:
             for d in devices:
                 if isinstance(d, dict) and d.get("serial"):
                     current_device_map[d.get("serial")] = d
        
        current_serials = set(current_device_map.keys())
        logger.info(f"Identified {len(current_serials)} unique valid devices.")

        # 2. Init Zabbix Service
        try:
            logger.info("Initializing Zabbix service and fetching host groups...")
            zabbix = ZabbixService()
            zabbix_groups = zabbix.get_hostgroups() # {name: id}
            logger.info(f"Fetched {len(zabbix_groups)} host groups from Zabbix.")
        except Exception as e:
            logger.error(f"Failed to initialize Zabbix service: {e}")
            return JsonResponse({"error": f"Zabbix connection failed: {str(e)}"}, status=500)
        
        # 3. Redis & Sync Logic
        # Get cached serials (previous state)
        cached_serials_list = get_json(ZABBIX_SYNCED_HOSTS_KEY) or []
        cached_serials = set(cached_serials_list)
        logger.info(f"Loaded {len(cached_serials)} cached serials from Redis.")

        # Calculate Diff
        to_disable = cached_serials - current_serials
        # to_add_or_update = current_serials # We process all current devices to ensure they exist
        logger.info(f"Diff calculation: {len(to_disable)} hosts to disable, {len(current_serials)} hosts to add/update.")

        results = {
            "created": [],
            "skipped": [],
            "failed": [],
            "disabled": [],
            "failed_disable": []
        }

        # 3.1 Disable removed hosts
        if to_disable:
            logger.info(f"Processing {len(to_disable)} disabled hosts with {MAX_WORKERS} threads...")
            
            def _disable_task(s_serial):
                # Prepare update parameters
                desc = "该设备从meraki设备接口 无法发现, 设备已经从meraki平台删除!"
                tags = None
                if disable_host_tag and isinstance(disable_host_tag, dict):
                    # Note: This replaces existing tags. If we want to append, we'd need to fetch first.
                    # But for disabled hosts, replacing or setting a specific tag is usually fine.
                    # Zabbix API requires tags to be a list of objects.
                    tags = [disable_host_tag]
                
                # Update host details (Status=1 means disable)
                return zabbix.update_host_details(
                    host_name=s_serial,
                    description=desc,
                    tags=tags,
                    status=1
                )

            with concurrent.futures.ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
                future_to_serial = {executor.submit(_disable_task, s): s for s in to_disable}
                for future in concurrent.futures.as_completed(future_to_serial):
                    s = future_to_serial[future]
                    try:
                        res = future.result()
                        # update_host_details returns "updated" on success
                        if res["status"] == "updated":
                            results["disabled"].append(res)
                        elif res["status"] == "not_found":
                             # It might have been deleted manually
                             pass
                        else:
                            results["failed_disable"].append(res)
                            logger.warning(f"Failed to disable host {s}: {res}")
                    except Exception as exc:
                        logger.error(f"Disable task generated an exception for {s}: {exc}")
                        results["failed_disable"].append({"host": s, "error": str(exc)})

            logger.info(f"Disabled processing complete. Success: {len(results['disabled'])}, Failed: {len(results['failed_disable'])}")

        # Pre-fetch Template IDs
        logger.info("Pre-fetching Zabbix template IDs...")
        template_cache = {}
        # Added Env_Meraki_ICMP_Ping to the list
        for tmpl_name in ["Env_Meraki_Wireless_Template", "Env_Meraki_Switch_Template", "Env_Meraki_ICMP_Ping"]:
             tid = zabbix.get_template_id_by_name(tmpl_name)
             if tid:
                 template_cache[tmpl_name] = tid
                 logger.info(f"Template '{tmpl_name}' found, ID: {tid}")
             else:
                 logger.warning(f"Template not found: {tmpl_name}")
        
        # 3.2 Process Current Devices (Add/Update)
        # Filter valid devices first (wireless/switch only)
        valid_devices_to_sync = []
        for serial, dev in current_device_map.items():
            p_type = dev.get("productType")
            if p_type not in ["wireless", "switch"]:
                # Skip non-supported types
                continue
            valid_devices_to_sync.append(dev)
            
        logger.info(f"Processing {len(valid_devices_to_sync)} valid devices (wireless/switch) with {MAX_WORKERS} threads...")

        def _create_update_task(dev_info):
            serial = dev_info.get("serial")
            name = dev_info.get("name") # Visible Name
            product_type = dev_info.get("productType")
            network_name = dev_info.get("networkName")
            meraki_tags = dev_info.get("tags") or []
            
            # Determine IP (Priority: lanIp -> publicIp -> None)
            ip = dev_info.get("lanIp")
            if not ip:
                ip = dev_info.get("publicIp")
            
            # Construct SNMP Interface
            interfaces = []
            if ip:
                interfaces.append({
                    "type": 2, # SNMP
                    "main": 1,
                    "useip": 1,
                    "ip": ip,
                    "dns": "",
                    "port": "161",
                    "details": {
                        "version": 2,
                        "community": "{$SNMP_COMMUNITY}"
                    }
                })
            else:
                 # Without IP, we use 127.0.0.1 placeholder for SNMP interface
                 interfaces.append({
                    "type": 2,
                    "main": 1,
                    "useip": 1,
                    "ip": "127.0.0.1",
                    "dns": "",
                    "port": "161",
                    "details": {
                        "version": 2,
                        "community": "{$SNMP_COMMUNITY}"
                    }
                })

            # Determine Target Group
            target_group_name = None
            
            # Priority 1: Map
            if network_name and network_name in network_group_map:
                target_group_name = network_group_map[network_name]
            
            # Priority 2: Exact Match in Zabbix (if networkName exists as a Zabbix Group)
            if not target_group_name and network_name and network_name in zabbix_groups:
                target_group_name = network_name

            # Priority 3: Default Rule
            if not target_group_name:
                target_group_name = get_default_zabbix_group_name(product_type)

            # Resolve Group ID
            group_id = zabbix_groups.get(target_group_name)
            
            if not group_id:
                return {
                    "status": "failed",
                    "host": serial,
                    "name": name,
                    "reason": f"Target group '{target_group_name}' not found in Zabbix"
                }

            # Resolve Template ID
            templates = []
            target_template_name = None
            if product_type == "wireless":
                target_template_name = "Env_Meraki_Wireless_Template"
            elif product_type == "switch":
                target_template_name = "Env_Meraki_Switch_Template"
            
            if target_template_name and target_template_name in template_cache:
                templates.append({"templateid": template_cache[target_template_name]})
            
            # Always add ICMP Ping template
            if "Env_Meraki_ICMP_Ping" in template_cache:
                templates.append({"templateid": template_cache["Env_Meraki_ICMP_Ping"]})

            # Macros
            macros = [
                {"macro": "{$SERIAL}", "value": serial or ""},
                {"macro": "{$TYPE}", "value": product_type or ""},
                {"macro": "{$SNMP_COMMUNITY}", "value": "public"} # Default community
            ]
            
            if meraki_url_base:
                 macros.append({"macro": "{$MERAKI_URL}", "value": meraki_url_base})

            # Tags
            tags = []
            for t in meraki_tags:
                tags.append({"tag": "meraki", "value": t})

            # Create Host
            res = zabbix.create_host(serial, name, group_id, interfaces, templates=templates, macros=macros, tags=tags)
            
            # Inject group info into result for logging/response
            res["group"] = target_group_name
            return res

        with concurrent.futures.ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
            future_to_dev = {executor.submit(_create_update_task, d): d for d in valid_devices_to_sync}
            for future in concurrent.futures.as_completed(future_to_dev):
                d_info = future_to_dev[future]
                s_serial = d_info.get("serial")
                s_name = d_info.get("name")
                try:
                    res = future.result()
                    if res["status"] == "created":
                        results["created"].append(res)
                    elif res["status"] in ["skipped", "updated_group"]:
                        results["skipped"].append(res)
                    else:
                        results["failed"].append(res)
                        logger.error(f"Failed to create/update host {s_serial}: {res}")
                except Exception as exc:
                    logger.error(f"Create/Update task generated an exception for {s_serial}: {exc}")
                    results["failed"].append({"host": s_serial, "name": s_name, "error": str(exc)})

        logger.info(f"Sync complete. Created: {len(results['created'])}, Skipped/Updated: {len(results['skipped'])}, Failed: {len(results['failed'])}")

        # 4. Update Cache & Backup
        # Only update cache if we successfully processed at least some devices or if we intended to clear
        # But generally, we update cache to reflect the current state (current_serials)
        # Exception: if current_serials is empty but we had cached_serials, it means all were disabled.
        
        new_synced_list = list(current_serials)
        set_json(ZABBIX_SYNCED_HOSTS_KEY, new_synced_list) # No expire (permanent)
        logger.info("Updated Redis sync cache.")

        # Backup to text file
        try:
            log_dir = settings.LOG_DIR
            backup_file = os.path.join(log_dir, "zabbix_synced_hosts.txt")
            with open(backup_file, "w", encoding="utf-8") as f:
                for s in new_synced_list:
                    f.write(f"{s}\n")
            logger.info(f"Backed up synced hosts list to {backup_file}")
        except Exception as e:
            logger.error(f"Failed to backup synced hosts to file: {e}")

        return JsonResponse(results, safe=False, json_dumps_params={'indent': 2, 'ensure_ascii': False})

    except Exception as e:
        logger.exception("sync_meraki_to_zabbix failed")
        return JsonResponse({"error": str(e)}, status=500)
